/*=====================================================================
  Project:   Netrix.bdsproj

  File:      Class.cs

  Summary:   By recreating a popular game, this simple project
             demonstrates how to use System.Drawing capabilities
             of .NET Framework, and how to handle keyboard key
             strokes in your .NET application.

---------------------------------------------------------------------
	 Example submitted by Alfred Mirzagitov

=====================================================================*/

using System;
using System.Drawing;
using System.Windows.Forms;

namespace Netrix
{
  /// <summary>
  /// Summary description for Class.
  /// </summary>

  public enum MyColors
  {
    Red     = 1,
    Blue    = 2,
    Orange  = 3,
    Yellow  = 4,
    Lime    = 5,
    Aqua    = 6,
    Magenta = 7,
    Black   = 8
  }

  public class PlayingField: System.Windows.Forms.Panel
  {
    public int[ , ] pfmatrix; //col, row

    private WinForm FOwner;
    private GamePiece gp;
    private Timer MainTimer;
    private int RowsRemoved = 0;
    private int CurrentDelay = 500;

    public int FieldWidth = 10;
    public int FieldHeight = 20;

    public PlayingField(WinForm aOwner, int x, int y, int h, int w)
    {
      FOwner = aOwner;

      this.Parent = FOwner;
      this.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom)
                    | System.Windows.Forms.AnchorStyles.Right)));
      this.BorderStyle = System.Windows.Forms.BorderStyle.None;
      this.Location = new System.Drawing.Point(x, y);
      this.Name = "MainPanel";
      this.Size = new System.Drawing.Size(h, w);
      this.TabIndex = 3;

      pfmatrix = new int[FieldWidth, FieldHeight];
      EmptyPlayingField();
      gp = new GamePiece(this);

      MainTimer = new Timer();
      MainTimer.Tick += new System.EventHandler(this.TimerEvent);
    }

    protected override void OnPaint(PaintEventArgs e)
    {
      Graphics g = e.Graphics;

      SuspendLayout();

      SolidBrush b = new SolidBrush(Color.FromArgb(180, Color.White));

      int x, hor_offset, ver_offset, current;
      if ((Width / FieldWidth) < (Height / FieldHeight))
        x = Width / FieldWidth;
      else
        x = Height / FieldHeight;

      hor_offset = (Width - x * FieldWidth) / 2;
      ver_offset = (Height - x * FieldHeight) / 2;

      Pen p = new Pen(Color.Gray);
      g.DrawLine(p, hor_offset,
                    ver_offset,
                    hor_offset,
                    ver_offset+x * FieldHeight);
      g.DrawLine(p, hor_offset,
                    ver_offset+x * FieldHeight,
                    hor_offset+x * FieldWidth,
                    ver_offset+x * FieldHeight);
      g.DrawLine(p, hor_offset+x * FieldWidth,
                    ver_offset+x * FieldHeight,
                    hor_offset+x * FieldWidth,
                    ver_offset);

      x = x - 1;

      for (int r = 0; r < FieldHeight; r++)
        for (int c = 0; c < FieldWidth; c++) {
          current = pfmatrix[c,r];

          if (gp != null) {
            if ((gp.row <= r) && (r < gp.row+gp.cMaxRows) &&
                (gp.col <= c) && (c < gp.col+gp.cMaxCols))
                current |= gp.cPiece[c-gp.col,r-gp.row];
          }

          switch (current) {
            case (int)MyColors.Red:     b.Color = Color.Red;     break;
            case (int)MyColors.Lime:    b.Color = Color.Lime;    break;
            case (int)MyColors.Orange:  b.Color = Color.Orange;  break;
            case (int)MyColors.Blue:    b.Color = Color.Blue;    break;
            case (int)MyColors.Yellow:  b.Color = Color.Yellow;  break;
            case (int)MyColors.Black:   b.Color = Color.Black;   break;
            case (int)MyColors.Magenta: b.Color = Color.Magenta; break;
            case (int)MyColors.Aqua:    b.Color = Color.Aqua;    break;
            default:                    b.Color = BackColor;     break;
          }
          g.FillRectangle(b, hor_offset+1+c*(x+1), ver_offset+1+(FieldHeight-1-r)*(x+1),x,x);
        }
      ResumeLayout(false);
    }

    public void EmptyPlayingField()
    {
      int row, col;

      for (row = 0; row < FieldHeight; row++)
        for (col = 0; col < FieldWidth; col++)
          pfmatrix[col,row] = 0;
    }

    private void SpeedUp(int r)
    {
      if (r > 10) return;
      RowsRemoved++;
      if ((RowsRemoved > 35) && (CurrentDelay > 450))
        CurrentDelay -= 50;
      if ((RowsRemoved > 55) && (CurrentDelay > 400))
        CurrentDelay -= 50;
      if ((RowsRemoved > 75) && (CurrentDelay > 350))
        CurrentDelay -= 50;
      if ((RowsRemoved > 85) && (CurrentDelay > 300))
        CurrentDelay -= 50;
      if ((RowsRemoved > 90) && (CurrentDelay > 250))
        CurrentDelay -= 50;
      if ((RowsRemoved > 95) && (CurrentDelay > 200))
        CurrentDelay -= 50;
      if ((RowsRemoved > 100) && (CurrentDelay > 200))
        CurrentDelay -= 50;
    }

    private void RemoveRow(int r)
    {
      int row, col;

      SpeedUp(r);
      FOwner.AddScore(10*(r+1));

      for (row = r; row < FieldHeight-1; row++)
        for (col = 0; col < FieldWidth; col++)
          pfmatrix[col,row] = pfmatrix[col,row+1];

      for (col = 0; col < FieldWidth; col++)
        pfmatrix[col,FieldHeight-1] = 0;
    }

    public void RemoveRows(int score)
    {
      bool hole;
      int c,r = 0;

      FOwner.AddScore(score);
      do {
        hole = false;
        for (c = 0; c < FieldWidth; c++)
          if (pfmatrix[c,r] == 0) hole = true;
        if (hole)
          r++;
        else
          RemoveRow(r);
      } while (r < FieldHeight);
    }

    public void GoLeft()
    {
      if (gp != null) gp.StepLeft();
    }

    public void GoRight()
    {
      if (gp != null) gp.StepRight();
    }

    public void TurnClockwise()
    {
      if (gp != null) gp.Rotate(true);
    }

    public void TurnCounterclockwise()
    {
      if (gp != null) gp.Rotate(false);
    }

    public void GoDown()
    {
      if (gp != null) gp.StepDown();
    }

    public void Drop()
    {
      if (gp != null) gp.DropDown();
    }

    private void TimerEvent(object sender, System.EventArgs e)
    {
      if (gp != null) gp.StepDown();
    }

    public void PauseGame()
    {
      MainTimer.Enabled = false;
    }

    public void ResumeGame()
    {
      MainTimer.Enabled = true;
    }

    public void GameOver()
    {
      MainTimer.Enabled = false;
      FOwner.GameOver();
    }

    public void NewGame()
    {
      EmptyPlayingField();
      RowsRemoved = 0;
      CurrentDelay = 500;
      Invalidate();
      gp.init();
      MainTimer.Interval = CurrentDelay;
      MainTimer.Enabled = true;
    }
  }

  public class GamePiece: System.Windows.Forms.Panel
  {
    private PlayingField FOwner;
    public int[,] cPiece, nPiece; //current, next
    private MyColors cColor, nColor;
    private int nMaxCols, nMaxRows;

    public int cMaxRows, cMaxCols;

    public int col, row;

    public GamePiece(PlayingField aOwner)
    {
      FOwner = aOwner;
      init();
      init();
    }

    public void init()
    {
      int i;

      cPiece = nPiece;
      cColor = nColor;
      cMaxCols = nMaxCols;
      cMaxRows = nMaxRows;

      col = FOwner.FieldWidth / 2 - 1;
      row = FOwner.FieldHeight;

      Random rdm = new Random(unchecked((int)DateTime.Now.Ticks));

      do {
        i = rdm.Next(1, 9);
        nColor = (MyColors) i;
      } while (cColor == nColor);

      switch (rdm.Next(1,8)) {
        case 1:
          //  WW
          //  WW
          nPiece = new int[2,2] {{i,i},{i,i}};
          nMaxCols = 2;
          nMaxRows = 2;
          break;
        case 2:
          //  W
          //  W
          //  W
          //  W
          nPiece = new int[1,4] {{i,i,i,i}};
          nMaxCols = 1;
          nMaxRows = 4;
          break;
        case 3:
          //  W
          //  W
          //  WW
          nPiece = new int [2,3] {{i,i,i},{0,0,i}};
          nMaxCols = 2;
          nMaxRows = 3;
          break;
        case 4:
          //   W
          //   W
          //  WW
          nPiece = new int [2,3] {{0,0,i},{i,i,i}};
          nMaxCols = 2;
          nMaxRows = 3;
          break;
        case 5:
          //  W
          //  WW
          //   W
          nPiece = new int [2,3] {{i,i,0},{0,i,i}};
          nMaxCols = 2;
          nMaxRows = 3;
          break;
        case 6:
          //   W
          //  WW
          //  W
          nPiece = new int [2,3] {{0,i,i},{i,i,0}};
          nMaxCols = 2;
          nMaxRows = 3;
          break;
        default:
          //   W
          //  WW
          //   W
          nPiece = new int [2,3] {{0,i,0},{i,i,i}};
          nMaxCols = 2;
          nMaxRows = 3;
          break;
      }
    }

    public void Rotate(bool clockwise)
    {
      int c,r;
      int[,] xPiece = new int[cMaxRows, cMaxCols];

      if (clockwise) {
        for (c=0; c<cMaxCols; c++)
          for (r=0; r<cMaxRows; r++)
            xPiece[cMaxRows-1-r,c] = cPiece[c,r];
      }
      else {
        for (c=0; c<cMaxCols; c++)
          for (r=0; r<cMaxRows; r++)
            xPiece[r,cMaxCols-1-c] = cPiece[c,r];
      }
      if (!Overlap(xPiece, cMaxRows, cMaxCols))
      {
        cPiece = xPiece;
        c = cMaxRows;
        cMaxRows = cMaxCols;
        cMaxCols = c;
        FOwner.Invalidate();
      }
    }

    public bool Overlap(int[,] piece, int MaxCols, int MaxRows)
    {
      int c;

      if ((col < 0) || (row < 0)) return true;
      if (col + MaxCols > FOwner.FieldWidth) return true;

      for (int i = 0; i < MaxCols; i++)
        for (int j = 0; j < MaxRows; j++)
          if (row + j < FOwner.FieldHeight) {
            c = piece[i,j];
            if ((c>0) && (FOwner.pfmatrix[i+col,j+row]>0)) return true;
          }
      return false;
    }

    public void DropDown()
    {
      do {
        row--;
      } while (!Overlap(cPiece, cMaxCols, cMaxRows));
      row++;
      ConsolidatePiece();
      FOwner.Invalidate();
    }

    public void StepDown()
    {
      row--;
      if (Overlap(cPiece, cMaxCols, cMaxRows))
      {
        row++;
        ConsolidatePiece();
      }
      FOwner.Invalidate();
    }

    public void StepLeft()
    {
      col--;
      if (Overlap(cPiece, cMaxCols, cMaxRows))
        col++;
      else
        FOwner.Invalidate();
    }

    public void StepRight()
    {
      col++;
      if (Overlap(cPiece, cMaxCols, cMaxRows))
        col--;
      else
        FOwner.Invalidate();
    }

    public void ConsolidatePiece()
    {
      int count = 0;

      if (row + cMaxRows > FOwner.FieldHeight)
      {
        FOwner.GameOver();
        return;
      }

      for (int c = 0; c < FOwner.FieldWidth; c++)
        for (int r = 0; r < FOwner.FieldHeight; r++)
          if ((row <= r) && (r < row+cMaxRows) &&
              (col <= c) && (c < col+cMaxCols))
              {
                FOwner.pfmatrix[c,r] |= cPiece[c-col,r-row];
                if (cPiece[c-col,r-row] != 0) count++;
              }

      FOwner.RemoveRows(count);
      FOwner.Invalidate();

      init();
    }
  }
}
